#!/usr/bin/perl -w
#
# vmstats-daemon.pl v0.02 by Stuart J. Browne
# Email: bekar@bekar.id.au
#
# Quick routine which runs as a daemon to gather stats from 'vmstat',
# and then dumps them out in an MRTG format upon request.
#
# Designed to work with vmstat 2.0.17, but easily modified for any other
# version (regex on line 125 or so).
#

use strict;
use Sys::Syslog;
use Sys::Hostname;
use POSIX ();

my $pid_file	= "/var/run/vmstat.pid";
my $dump_file	= "/tmp/vmstats.out";
my $vmstat_bin	= "/usr/bin/vmstat";
my $vmstat_args	= "-ban";
my $uptime_bin	= "/usr/bin/uptime";
my $dump_period	= 15;

#
# No further user-definable bits beyond here!
#

my $signal_set	= 0;
my $loop_count	= 0;

&openlog( "vmstat_log", "pid", "local0" );

#
# Make a daemon out of me
#
sub daemonize {
	chdir ('/')			or die "Can't chdir to /: $!";
	open  ( STDIN, "/dev/null" )	or die "Can't read /dev/null: $!";
	open  ( STDOUT, ">/dev/null" )	or die "Can't write to /dev/null: $!";
	defined(my $pid = fork)		or die "Can't fork: $!";
	exit if $pid;
	&POSIX::setsid			or die "Can't start a new session: $!";
	open  ( STDERR, ">&STDOUT" )	or die "Can't dup stdout: $!";

	open  ( PIDFILE, "> " . $pid_file );
	printf( PIDFILE "%d\n", $$ );
	close ( PIDFILE );

	$SIG{HUP}	= 'IGNORE';
	$SIG{TERM}	= \&kill_me;
}

#
# Received TERM signal.  Remove PID file, and die!
#
sub kill_me {
	my $kill_msg = shift;

	if ( $kill_msg =~ /TERM/ ) {
		$kill_msg = "Requested termination.  Terminating now." ;
	}

	close( STATS );
	unlink( $pid_file );
	unlink( $dump_file );
	&syslog( "notice", $kill_msg );
	&closelog;
	die "Requested termination.  Terminating now.";
}

#
# Received HUP signal.
#
sub dump_me {
	$signal_set	= 1;
	$SIG{HUP}	= 'IGNORE';
}

#
# Open up temporary file, dump statistics
#
sub dump_stats {
	&syslog( "debug", "Stats Dumping Requested.  Dumping stats now (%ds).  Ignoring HUP signal for %ds.", $loop_count, $dump_period );
	my ( $s, $f, $i, $a, $si, $so, $bi, $bo ) = @_;
	open( OUT_FILE, "> " . $dump_file );
	printf( OUT_FILE "%d\t%d\t%d\t%d\t%d\t%d\t%d\t%d\n", $s, $f, $i, $a, $si, $so, $bi, $bo );
	close( OUT_FILE );
	$signal_set = 0;
};

#
# Have been called with '-d', set up as daemon, and loop gatherting stats every second.
#
sub gather_stats {
	-x $vmstat_bin or die( "Couldn't find vmstat binary where expected.  Please check \$vmstat_bin in configuration." );
	&daemonize and &syslog( "notice", "VMStats logger started.  Successfully daemonized." );
	while (1) {
		open( STATS, sprintf( "%s %s 1 %s", $vmstat_bin, $vmstat_args, " 2>&1 |" ) )	or die "Can't fork: $!";
#
# Our stats look like (2.0.17):
#  procs                              memory      swap          io     system         cpu
#  r  b     swpd     free    inact    active   si   so    bi    bo   in    cs us sy wa id
#  0  0 46751744 26660864 93413376 212750336    0    1     6     0    6     6  1  0  0  3
#

		my $swapd	= 0;
		my $free	= 0;
		my $inact	= 0;
		my $active	= 0;
		my $swap_in	= 0;
		my $swap_out	= 0;
		my $blocks_in	= 0;
		my $blocks_out	= 0;

		while (<STATS>) {
			$loop_count++;
#
# This is the beasty that drives it all.  2.0.17 version of the line here.
#
# if (/^\s*(?:\d+\s+){2}(\d+)\s+(\d+)\s+(\d+)\s+(\d+)\s+(\d+)\s+(\d+)\s+(\d+)\s+(\d+)[\s\d]+$/) {
#
# This is a 2.0.11 version:
#
# if (/^\s*(?:\d+\s+){3}(\d+)\s+(\d+)\s+(\d+)\s+(\d+)\s+(\d+)\s+(\d+)\s+(\d+)\s+(\d+)[\s\d]+$/) {
#
			if (/^\s*(?:\d+\s+){2}(\d+)\s+(\d+)\s+(\d+)\s+(\d+)\s+(\d+)\s+(\d+)\s+(\d+)\s+(\d+)[\s\d]+$/) {
				$swapd		=  $1;
				$free		=  $2;
				$inact		=  $3;
				$active		=  $4;
				$swap_in	+= $6;
				$swap_out	+= $5;
				$blocks_in	+= $8;
				$blocks_out	+= $7;
			} else {
				next;
			};
			if ($signal_set == 1) {
				&dump_stats( $swapd, $free, $inact, $active, $swap_in, $swap_out, $blocks_in, $blocks_out );
				$swapd		= 0;
				$free		= 0;
				$inact		= 0;
				$active		= 0;
				$swap_in	= 0;
				$swap_out	= 0;
				$blocks_in	= 0;
				$blocks_out	= 0;
				$loop_count	= 0;
			}
			if ( ( $loop_count > $dump_period ) && ( $loop_count < $dump_period + 5) ) {
				$SIG{HUP}	= \&dump_me;
			}
		}
		close(STATS);
		&syslog( "notice", "Child process died.  Attempting to restart it." );
	}
	&kill_me( sprintf( "Child '%s' process died.  Terminating now.", $vmstat_bin ) );
}

#
# Actually dump out the stats to STDOUT for MRTG
#
sub print_stats {
	-f $pid_file or die( "VMStat collection daemon not running." );

	open( PIDFILE, $pid_file );
	my $pid = <PIDFILE>;
	close( PIDFILE );
	kill HUP => $pid or die "Couldn't get new stats from daemon..";
	sleep( 2 );

	-f $dump_file or die( "Dumped statistics couldn't be found." );
	-x $uptime_bin or die( "Couldn't find uptime binary where expected.  Please check \$uptime_bin in configuration." );

	my $action	= shift;
	my $uptime	= '';
	my $in		= 0;
	my $out		= 0;

	open ( UPTIME, $uptime_bin . " |" );
	map { $uptime = $1 if ( m/^.*up\s+(.*),\s+[0-9]+\s+[Uu]ser.*$/ ); } <UPTIME>;
	close UPTIME;

	open( STAT_FILE, $dump_file );

	if ( $action =~ /f/ ) {
		map { ( $in, $out ) = ( $1, $2 ) if ( m/^([0-9]+)\s+([0-9]+).*$/ ); } <STAT_FILE>;
	} elsif ( $action =~ /m/ ) {
		map { ( $in, $out ) = ( $2, $1 ) if ( m/^(?:[0-9]+\s+){2}([0-9]+)\s+([0-9]+).*$/ ); } <STAT_FILE>;
	} elsif ( $action =~ /s/ ) {
		map { ( $in, $out ) = ( $1, $2 ) if ( m/^(?:[0-9]+\s+){4}([0-9]+)\s+([0-9]+).*$/ ); } <STAT_FILE>;
	} elsif ( $action =~ /b/ ) {
		map { ( $in, $out ) = ( $1, $2 ) if ( m/^(?:[0-9]+\s+){6}([0-9]+)\s+([0-9]+)$/ ); } <STAT_FILE>;
	}

	close( STAT_FILE );

	printf( "%d\n%d\n%s\n%s\n", $in, $out, &hostname, $uptime );
}

#
# Basic helper screen
#
sub help_screen {
	print << "EOF"

Simple statistics gatherer for MRTG graphing from the output of 'vmstat'.
Usage:
	Start background daemon:
		vmstat-daemon.pl -d	

	Collect statistics:
		vmstat-daemon.pl -f		# Allocated/Free Swap
		vmstat-daemon.pl -m		# Inactive/Active memory pages
		vmstat-daemon.pl -s		# Swap blocks in/out
		vmstat-daemon.pl -b		# IO blocks in/out

EOF
}

#
# Main part, get arguments
#
my $my_arg = shift;

if ( defined( $my_arg ) ) {
	if ( $my_arg =~ /^-d$/ ) {
		print "I'm called in daemon mode.  Attempting to fork.\n";
		&gather_stats or die "I wasn't able to. Sorry!";
	} elsif ( $my_arg =~ /^-m$/ ) {
		&print_stats( "m" );
	} elsif ( $my_arg =~ /^-b$/ ) {
		&print_stats( "b" );
	} elsif ( $my_arg =~ /^-s$/ ) {
		&print_stats( "s" );
	} elsif ( $my_arg =~ /^-f$/ ) {
		&print_stats( "f" );
	} else {
		printf( "Called with invalid arguments.\n" );
		&help_screen;
	}
} else {
#
# No arguments, give helper screen
#
	printf( "Called with no arguments.\n" );
	&help_screen;
}
